type RuleFunction = (value: any, label: string, param?: any) => true | string;

export function useValidate<T = any>(props: Record<string, any>, model: Ref<T>) {

    const error = ref<boolean | string>(false);
    const isValid=ref<boolean>(false);

    const functions: Record<string,RuleFunction>={
        min(value, label, length) {
            if (value.toString().trim().length < +length) {
                return `${label} حداقل باید دارای ${length} کاراکتر باشد`;
            }
            return true;
        },
        max(value, label, length) {
            if (value.toString().trim().length > +length) {
                return `${label} حداکثر باید دارای ${length} کاراکتر باشد`;
            }
            return true;
        },
        nationalCode(value, label) {
            if (!isNaN(parseInt(value)) && value.toString().length === 10) {
                let sum = 0;
                for (let i = 0; i < 9; i++) {
                    sum += (10 - i) * parseInt(value[i]);
                }
                const ret = sum % 11;
                const parity = parseInt(value[9]);
                if ((ret < 2 && ret === parity) || (ret >= 2 && ret === 11 - parity)) {
                    return true;
                }
            }
            return `${label} معتبر نمی باشد`;
        },
        required(value, label) {
            if (value == null || value.toString().trim() === '') {
                return `${label} نمی تواند خالی باشد`;
            }
            return true;
        },
        mobile(value, label) {
            if (!isNaN(value)) {
                value = parseInt(value);
                if (value.toString().length === 10 && value.toString().charAt(0) === '9') {
                    return true;
                }
            }
            return `${label} وارد شده معتبر نمی باشد`;
        },
        length(value, label, length) {
            if (value.toString().trim().length !== +length) {
                return `${label} باید شامل ${length} کاراکتر باشد`;
            }
            return true;
        },
        email(value, label) {
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!value.toString().trim().match(emailRegex)) {
                return `${label} باید یک ایمیل معتبر باشد`;
            }
            return true;
        },
    };

    const validate = (value: any, rules: string | RuleFunction[], label: string): true | string => {
        const parsedRules: (string | RuleFunction)[] =
            typeof rules === 'string' ? rules.split('|') : rules;

        for (const rule of parsedRules) {
            let result: true | string = true;

            if (typeof rule === 'function') {
                result = rule(value, label);
            } else {
                const [name, param] = rule.split(':');
                const fn = functions[name];
                if (fn) {
                    result = fn(value, label, param);
                }
            }

            if (result !== true) {
                return result;
            }
        }

        return true;
    };

    const validateInput = (): true | string => {
        let result: true | string = true;
        if (props.rules) {
            result = validate(model.value, props.rules, props.label);
        }
        error.value = result === true ? false : result;
        isValid.value=result === true ;
        return result;
    };

    const inputKeyUp = () => {
        validateInput();
    };

    const checknationalCode = (value: string, label: string): true | string => {
        if (!isNaN(parseInt(value)) && value.length === 10) {
            let sum = 0;
            for (let i = 0; i < 9; i++) {
                sum += (10 - i) * parseInt(value[i]);
            }
            const ret = sum % 11;
            const parity = parseInt(value[9]);
            if ((ret < 2 && ret === parity) || (ret >= 2 && ret === 11 - parity)) {
                return true;
            }
        }
        return `${label} معتبر نمی باشد`;
    };

    return {
        error,
        isValid,
        functions,
        inputKeyUp,
        validateInput,
        checknationalCode,
    };
}
