import { readFile } from 'node:fs/promises';
import { resolve, relative, basename } from 'node:path';
import process from 'node:process';
import { instructions } from '@vite-pwa/assets-generator/api/instructions';
import { loadConfig } from '@vite-pwa/assets-generator/config';
import { p as pwaIcons, g as generatePwaImageType } from '../shared/nuxt.689bb9f6.mjs';
import '@nuxt/kit';
import 'vite-plugin-pwa';
import 'node:crypto';
import 'node:fs';
import 'pathe';

async function preparePWAIconTypes(options, nuxt) {
  if (!options.pwaAssets || options.pwaAssets.disabled)
    return;
  const configuration = resolvePWAAssetsOptions(options);
  if (!configuration || configuration.disabled)
    return;
  const root = nuxt.options.vite.root ?? process.cwd();
  const { config, sources } = await loadConfiguration(root, configuration);
  if (!config.preset)
    return;
  const {
    preset,
    images,
    headLinkOptions: userHeadLinkOptions
  } = config;
  if (!images)
    return;
  if (Array.isArray(images) && (!images.length || images.length > 1))
    return;
  const useImage = Array.isArray(images) ? images[0] : images;
  const imageFile = resolve(root, useImage);
  const publicDir = resolve(root, nuxt.options.dir.public ?? "public");
  const imageName = relative(publicDir, imageFile);
  const xhtml = userHeadLinkOptions?.xhtml === true;
  const includeId = userHeadLinkOptions?.includeId === true;
  const assetsInstructions = await instructions({
    imageResolver: () => readFile(resolve(root, useImage)),
    imageName,
    preset,
    faviconPreset: userHeadLinkOptions?.preset,
    htmlLinks: { xhtml, includeId },
    basePath: nuxt.options.app.baseURL ?? "/",
    resolveSvgName: userHeadLinkOptions?.resolveSvgName ?? ((name) => basename(name))
  });
  const transparentNames = Object.values(assetsInstructions.transparent).map(({ name }) => name);
  const maskableNames = Object.values(assetsInstructions.maskable).map(({ name }) => name);
  const faviconNames = Object.values(assetsInstructions.favicon).map(({ name }) => name);
  const appleNames = Object.values(assetsInstructions.apple).map(({ name }) => name);
  const appleSplashScreenNames = Object.values(assetsInstructions.appleSplashScreen).map(({ name }) => name);
  const dts = {
    dts: pwaIcons({
      transparent: transparentNames,
      maskable: maskableNames,
      favicon: faviconNames,
      apple: appleNames,
      appleSplashScreen: appleSplashScreenNames
    }),
    transparent: generatePwaImageType("PwaTransparentImage", transparentNames),
    maskable: generatePwaImageType("PwaMaskableImage", maskableNames),
    favicon: generatePwaImageType("PwaFaviconImage", faviconNames),
    apple: generatePwaImageType("PwaAppleImage", appleNames),
    appleSplashScreen: generatePwaImageType("PwaAppleSplashScreenImage", appleSplashScreenNames)
  };
  if (nuxt.options.dev && nuxt.options.ssr) {
    sources.forEach((source) => nuxt.options.watch.push(source.replace(/\\/g, "/")));
  }
  return dts;
}
function resolvePWAAssetsOptions(options) {
  if (!options.pwaAssets)
    return;
  const {
    disabled: useDisabled,
    config,
    preset,
    image = "public/favicon.svg",
    htmlPreset = "2023",
    overrideManifestIcons = false,
    includeHtmlHeadLinks = true,
    injectThemeColor = true,
    integration
  } = options.pwaAssets ?? {};
  const disabled = useDisabled || !config && !preset;
  return {
    disabled,
    config: disabled || !config ? false : config,
    preset: disabled || config ? false : preset ?? "minimal-2023",
    images: [image],
    htmlPreset,
    overrideManifestIcons,
    includeHtmlHeadLinks,
    injectThemeColor,
    integration
  };
}
async function loadConfiguration(root, pwaAssets) {
  if (pwaAssets.config === false) {
    return await loadConfig(root, {
      config: false,
      preset: pwaAssets.preset,
      images: pwaAssets.images,
      logLevel: "silent"
    });
  }
  return await loadConfig(
    root,
    typeof pwaAssets.config === "boolean" ? root : { config: pwaAssets.config }
  );
}

export { preparePWAIconTypes };
